from flask import Blueprint, jsonify, request
from datetime import datetime
from ..database import get_db_connection, handle_db_error
from .. import limiter

bp = Blueprint('bills', __name__, url_prefix='/api')

@bp.route('/bills')
@limiter.limit("30 per minute")  # More restrictive for list endpoint
def get_bills():
    try:
        # Get query parameters
        page = int(request.args.get('page', 1))
        per_page = min(int(request.args.get('per_page', 50)), 100)  # Cap at 100
        offset = (page - 1) * per_page
        
        conn = get_db_connection()
        cursor = conn.cursor(dictionary=True)
        
        # Get total count
        cursor.execute("SELECT COUNT(*) as count FROM bill_listings_exp")
        total_count = cursor.fetchone()['count']
        
        # Get bills with pagination
        query = """
            SELECT * FROM bill_listings_exp 
            ORDER BY last_action_date DESC 
            LIMIT %s OFFSET %s
        """
        cursor.execute(query, (per_page, offset))
        bills = cursor.fetchall()
        
        # Convert datetime objects to strings
        for bill in bills:
            for key, value in bill.items():
                if isinstance(value, datetime):
                    bill[key] = value.isoformat()
        
        return jsonify({
            "total": total_count,
            "page": page,
            "per_page": per_page,
            "bills": bills
        })
        
    except Exception as e:
        return handle_db_error(e)
    finally:
        if 'cursor' in locals():
            cursor.close()
        if 'conn' in locals():
            conn.close()

@bp.route('/bills/<bill_id>')
@limiter.limit("60 per minute")  # Less restrictive for individual bills
def get_bill(bill_id):
    try:
        conn = get_db_connection()
        cursor = conn.cursor(dictionary=True)
        
        query = "SELECT * FROM bill_listings_exp WHERE bill_id = %s"
        cursor.execute(query, (bill_id,))
        bill = cursor.fetchone()
        
        if not bill:
            return jsonify({"error": "Bill not found"}), 404
            
        # Convert datetime objects to strings
        for key, value in bill.items():
            if isinstance(value, datetime):
                bill[key] = value.isoformat()
        
        return jsonify(bill)
        
    except Exception as e:
        return handle_db_error(e)
    finally:
        if 'cursor' in locals():
            cursor.close()
        if 'conn' in locals():
            conn.close() 